﻿using System;
using System.Web.Mvc;
using System.Linq;
using MVC_App.Models;
using MVC_App.Validation;

namespace MVC_App.Controllers
{
	public class HomeController : Controller
	{
		[HttpGet]
		public ActionResult EditOrder()
		{
			return View(BuildDummyBestellung());
		}

		static BestellungForm BuildDummyBestellung()
		{
			return new BestellungForm
					{
						Lieferadresse = new LieferadresseForm(),
						Kreditkarte = new KreditkarteForm(),
						Pizza = new PizzaForm
											{
												Größe = PizzaGrößen.Mittel,
											}
					};
		}

		[HttpPost]
		public ActionResult SubmitOrder(BestellungForm bestellung)
		{
			var lieferzeitpunkt = ValidateLieferzeitpunkt(bestellung);

			if (ModelState.IsValid && lieferzeitpunkt != null)
			{
				// TODO: Ergebnisse verarbeiten
				return View("Success");
			}

			return View("EditOrder", bestellung);
		}

		private DateTime? ValidateLieferzeitpunkt(BestellungForm bestellung)
		{
			FactoryTime time;
			switch (bestellung.LieferzeitpunktArt)
			{
				case Lieferzeitpunkte.Sofort:
					return DateTime.Now;

				case Lieferzeitpunkte.Heute:
					time = ValidateUhrzeit(bestellung);
					if (time == null)
						return null;

					return DateTime.Now.SetTime(time);

				case Lieferzeitpunkte.Am:
					time = ValidateUhrzeit(bestellung);
					var date = ValidateDatum(bestellung);

					if (time == null || date == null)
						return null;

					return date.Value.SetTime(time);

				default:
					return null;
			}
		}

		private FactoryTime ValidateUhrzeit(BestellungForm bestellung)
		{
			var zeit = bestellung.Lieferzeitpunkt.Datum;
			if (zeit != null)
			{
				return FactoryTime.Build(zeit);
			}

			ModelState.AddModelError("Lieferzeitpunkt.Uhrzeit", "Bitte geben Sie eine gültige Uhrzeit für die Lieferung an.");
			return null;
		}

		private DateTime? ValidateDatum(BestellungForm bestellung)
		{
			var datumString = bestellung.Lieferzeitpunkt.Datum;
			if (datumString != null)
			{
				DateTime datum;
				if (DateTime.TryParse(datumString, out datum))
				{
					return datum;
				}
			}

			ModelState.AddModelError("Lieferzeitpunkt.Datum", "Bitte geben Sie ein gültiges Datum für die Lieferung an.");
			return null;
		}
	}

	public class FactoryTime
	{
		private FactoryTime(int hour, int minute)
		{
			Hour = hour;
			Minute = minute;
		}

		public int Hour { get; private set; }
		public int Minute { get; private set; }

		public static FactoryTime Build(int minute, int hour)
		{
			if (minute >= 0 && minute <= 60
				&& hour >= 0 && hour <= 24)
			{
				return new FactoryTime(hour, minute);
			}
			return null;
		}

		public static FactoryTime Build(string timeString)
		{
			var parts = timeString.Split(':');
			if (parts.Length == 2)
			{
				int hour;
				int minute;
				if (Int32.TryParse(parts[0], out hour)
					&& Int32.TryParse(parts[1], out minute))
				{
					return Build(minute, hour);
				}
			}
			return null;
		}
	}

	static class DateTimeExSnippets
	{
		public static DateTime SetTime(this DateTime date, FactoryTime time)
		{
			return new DateTime(date.Year, date.Month, date.Day, time.Hour, time.Minute, 0);
		}
	}
}
